<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type');

require 'connection.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'status' => 'error',
        'message' => 'Only POST method allowed'
    ]);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);

$action = $input['action'] ?? '';
$institute_code = $input['institute_code'] ?? '';

if (!$institute_code) {
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => 'Institute code is required'
    ]);
    exit;
}

try {
    switch ($action) {
        case 'get_student_attendance':
            getStudentAttendance($conn, $input);
            break;
        
        case 'get_student_monthly_report':
            getStudentMonthlyReport($conn, $input);
            break;
        
        case 'get_student_datewise_history':
            getStudentDatewiseHistory($conn, $input);
            break;
        
        case 'get_division_attendance':
            getDivisionAttendance($conn, $input);
            break;
        
        case 'get_division_datewise_history':
            getDivisionDatewiseHistory($conn, $input);
            break;
        
        case 'get_division_monthwise_history':
            getDivisionMonthwiseHistory($conn, $input);
            break;
        
        case 'get_students_list':
            getStudentsList($conn, $input);
            break;
        
        case 'get_courses':
            getCourses($conn, $institute_code);
            break;
        
        case 'get_years':
            getYears($conn, $input);
            break;
        
        case 'get_divisions':
            getDivisions($conn, $input);
            break;
        
        default:
            http_response_code(400);
            echo json_encode([
                'status' => 'error',
                'message' => 'Invalid action'
            ]);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}

$conn->close();

function getStudentAttendance($conn, $input) {
    $enrollno = $input['enrollno'] ?? '';
    $month = $input['month'] ?? date('m');
    $year = $input['year'] ?? date('Y');
    
    if (!$enrollno) {
        throw new Exception('Enrollment number is required');
    }
    
    $start_date = "$year-$month-01";
    $end_date = date("Y-m-t", strtotime($start_date));
    
    $stmt = $conn->prepare("
        SELECT COUNT(DISTINCT DATE(scan_time)) as present_days
        FROM att_log
        WHERE enrollno = ? 
        AND DATE(scan_time) >= ? 
        AND DATE(scan_time) <= ?
    ");
    $stmt->bind_param("sss", $enrollno, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $present_days = $result['present_days'];
    $stmt->close();
    
    $working_days = 0;
    $current = strtotime($start_date);
    $end = strtotime($end_date);
    
    while ($current <= $end) {
        $day_of_week = date('w', $current);
        if ($day_of_week != 0) 
            $working_days++;
        }
        $current = strtotime('+1 day', $current);
    }
    
    $attendance_percentage = $working_days > 0 ? round(($present_days / $working_days) * 100, 2) : 0;
    
    $stmt = $conn->prepare("
        SELECT Name, roll_no, email, contact
        FROM student_master
        WHERE enrollno = ?
    ");
    $stmt->bind_param("s", $enrollno);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'enrollno' => $enrollno,
            'name' => $student['Name'] ?? '',
            'roll_no' => $student['roll_no'] ?? '',
            'month' => $month,
            'year' => $year,
            'present_days' => $present_days,
            'working_days' => $working_days,
            'attendance_percentage' => $attendance_percentage
        ]
    ]);
}

function getStudentMonthlyReport($conn, $input) {
    $enrollno = $input['enrollno'] ?? '';
    $year = $input['year'] ?? date('Y');
    
    if (!$enrollno) {
        throw new Exception('Enrollment number is required');
    }
    
    $monthly_data = [];
    
    for ($month = 1; $month <= 12; $month++) {
        $month_str = str_pad($month, 2, '0', STR_PAD_LEFT);
        $start_date = "$year-$month_str-01";
        $end_date = date("Y-m-t", strtotime($start_date));
        
        $stmt = $conn->prepare("
            SELECT COUNT(DISTINCT DATE(scan_time)) as present_days
            FROM att_log
            WHERE enrollno = ? 
            AND DATE(scan_time) >= ? 
            AND DATE(scan_time) <= ?
        ");
        $stmt->bind_param("sss", $enrollno, $start_date, $end_date);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $present_days = $result['present_days'];
        $stmt->close();
        
        $working_days = 0;
        $current = strtotime($start_date);
        $end = strtotime($end_date);
        
        while ($current <= $end) {
            $day_of_week = date('w', $current);
            if ($day_of_week != 0) {
                $working_days++;
            }
            $current = strtotime('+1 day', $current);
        }
        
        $attendance_percentage = $working_days > 0 ? round(($present_days / $working_days) * 100, 2) : 0;
        
        $monthly_data[] = [
            'month' => $month,
            'month_name' => date('F', mktime(0, 0, 0, $month, 1)),
            'present_days' => $present_days,
            'working_days' => $working_days,
            'attendance_percentage' => $attendance_percentage
        ];
    }
    
    $stmt = $conn->prepare("
        SELECT Name, roll_no, email, contact
        FROM student_master
        WHERE enrollno = ?
    ");
    $stmt->bind_param("s", $enrollno);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'enrollno' => $enrollno,
            'name' => $student['Name'] ?? '',
            'roll_no' => $student['roll_no'] ?? '',
            'year' => $year,
            'monthly_report' => $monthly_data
        ]
    ]);
}

function getStudentDatewiseHistory($conn, $input) {
    $enrollno = $input['enrollno'] ?? '';
    $month = $input['month'] ?? date('m');
    $year = $input['year'] ?? date('Y');
    
    if (!$enrollno) {
        throw new Exception('Enrollment number is required');
    }
    
    $start_date = "$year-$month-01";
    $end_date = date("Y-m-t", strtotime($start_date));
    
    $stmt = $conn->prepare("
        SELECT 
            DATE(scan_time) as date,
            MIN(TIME(scan_time)) as first_in,
            MAX(TIME(scan_time)) as last_out,
            COUNT(*) as scan_count
        FROM att_log
        WHERE enrollno = ? 
        AND DATE(scan_time) >= ? 
        AND DATE(scan_time) <= ?
        GROUP BY DATE(scan_time)
        ORDER BY date ASC
    ");
    $stmt->bind_param("sss", $enrollno, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $attendance_history = [];
    while ($row = $result->fetch_assoc()) {
        $attendance_history[] = [
            'date' => $row['date'],
            'day_name' => date('l', strtotime($row['date'])),
            'first_in' => $row['first_in'],
            'last_out' => $row['last_out'],
            'scan_count' => $row['scan_count'],
            'status' => 'Present'
        ];
    }
    $stmt->close();
    
    $stmt = $conn->prepare("
        SELECT Name, roll_no
        FROM student_master
        WHERE enrollno = ?
    ");
    $stmt->bind_param("s", $enrollno);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'enrollno' => $enrollno,
            'name' => $student['Name'] ?? '',
            'roll_no' => $student['roll_no'] ?? '',
            'month' => $month,
            'year' => $year,
            'attendance_history' => $attendance_history
        ]
    ]);
}

function getDivisionAttendance($conn, $input) {
    $institute_code = $input['institute_code'] ?? '';
    $course_code = $input['course_code'] ?? '';
    $year_code = $input['year_code'] ?? '';
    $div_code = $input['div_code'] ?? '';
    $month = $input['month'] ?? date('m');
    $year = $input['year'] ?? date('Y');
    
    if (!$course_code || !$year_code || !$div_code) {
        throw new Exception('Course, year, and division codes are required');
    }
    
    $start_date = "$year-$month-01";
    $end_date = date("Y-m-t", strtotime($start_date));
    
    $working_days = 0;
    $current = strtotime($start_date);
    $end = strtotime($end_date);
    
    while ($current <= $end) {
        $day_of_week = date('w', $current);
        if ($day_of_week != 0) {
            $working_days++;
        }
        $current = strtotime('+1 day', $current);
    }
    
    $stmt = $conn->prepare("
        SELECT 
            sm.enrollno,
            sm.roll_no,
            sm.Name,
            COUNT(DISTINCT DATE(al.scan_time)) as present_days
        FROM student_master sm
        LEFT JOIN att_log al ON sm.enrollno = al.enrollno 
            AND DATE(al.scan_time) >= ? 
            AND DATE(al.scan_time) <= ?
        WHERE sm.institute_code = ? 
        AND sm.Course_Code = ? 
        AND sm.Year_Code = ? 
        AND sm.Div_code = ?
        GROUP BY sm.enrollno, sm.roll_no, sm.Name
        ORDER BY sm.roll_no ASC
    ");
    $stmt->bind_param("ssssss", $start_date, $end_date, $institute_code, $course_code, $year_code, $div_code);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $students = [];
    $total_present = 0;
    $student_count = 0;
    
    while ($row = $result->fetch_assoc()) {
        $present_days = $row['present_days'];
        $attendance_percentage = $working_days > 0 ? round(($present_days / $working_days) * 100, 2) : 0;
        
        $students[] = [
            'enrollno' => $row['enrollno'],
            'roll_no' => $row['roll_no'],
            'name' => $row['Name'],
            'present_days' => $present_days,
            'working_days' => $working_days,
            'attendance_percentage' => $attendance_percentage
        ];
        
        $total_present += $attendance_percentage;
        $student_count++;
    }
    $stmt->close();
    
    $division_average = $student_count > 0 ? round($total_present / $student_count, 2) : 0;
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'month' => $month,
            'year' => $year,
            'working_days' => $working_days,
            'total_students' => $student_count,
            'division_average' => $division_average,
            'students' => $students
        ]
    ]);
}

function getDivisionDatewiseHistory($conn, $input) {
    $institute_code = $input['institute_code'] ?? '';
    $course_code = $input['course_code'] ?? '';
    $year_code = $input['year_code'] ?? '';
    $div_code = $input['div_code'] ?? '';
    $month = $input['month'] ?? date('m');
    $year = $input['year'] ?? date('Y');
    
    if (!$course_code || !$year_code || !$div_code) {
        throw new Exception('Course, year, and division codes are required');
    }
    
    $start_date = "$year-$month-01";
    $end_date = date("Y-m-t", strtotime($start_date));
    
    $stmt = $conn->prepare("
        SELECT COUNT(*) as total_students
        FROM student_master
        WHERE institute_code = ? 
        AND Course_Code = ? 
        AND Year_Code = ? 
        AND Div_code = ?
    ");
    $stmt->bind_param("ssss", $institute_code, $course_code, $year_code, $div_code);
    $stmt->execute();
    $total_students = $stmt->get_result()->fetch_assoc()['total_students'];
    $stmt->close();
    
    $stmt = $conn->prepare("
        SELECT 
            DATE(al.scan_time) as date,
            COUNT(DISTINCT al.enrollno) as present_count
        FROM att_log al
        INNER JOIN student_master sm ON al.enrollno = sm.enrollno
        WHERE sm.institute_code = ? 
        AND sm.Course_Code = ? 
        AND sm.Year_Code = ? 
        AND sm.Div_code = ?
        AND DATE(al.scan_time) >= ? 
        AND DATE(al.scan_time) <= ?
        GROUP BY DATE(al.scan_time)
        ORDER BY date ASC
    ");
    $stmt->bind_param("ssssss", $institute_code, $course_code, $year_code, $div_code, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $datewise_history = [];
    while ($row = $result->fetch_assoc()) {
        $present_count = $row['present_count'];
        $attendance_percentage = $total_students > 0 ? round(($present_count / $total_students) * 100, 2) : 0;
        
        $datewise_history[] = [
            'date' => $row['date'],
            'day_name' => date('l', strtotime($row['date'])),
            'present_count' => $present_count,
            'total_students' => $total_students,
            'attendance_percentage' => $attendance_percentage
        ];
    }
    $stmt->close();
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'month' => $month,
            'year' => $year,
            'total_students' => $total_students,
            'datewise_history' => $datewise_history
        ]
    ]);
}

function getDivisionMonthwiseHistory($conn, $input) {
    $institute_code = $input['institute_code'] ?? '';
    $course_code = $input['course_code'] ?? '';
    $year_code = $input['year_code'] ?? '';
    $div_code = $input['div_code'] ?? '';
    $year = $input['year'] ?? date('Y');
    
    if (!$course_code || !$year_code || !$div_code) {
        throw new Exception('Course, year, and division codes are required');
    }
    
    $stmt = $conn->prepare("
        SELECT COUNT(*) as total_students
        FROM student_master
        WHERE institute_code = ? 
        AND Course_Code = ? 
        AND Year_Code = ? 
        AND Div_code = ?
    ");
    $stmt->bind_param("ssss", $institute_code, $course_code, $year_code, $div_code);
    $stmt->execute();
    $total_students = $stmt->get_result()->fetch_assoc()['total_students'];
    $stmt->close();
    
    $monthwise_data = [];
    
    for ($month = 1; $month <= 12; $month++) {
        $month_str = str_pad($month, 2, '0', STR_PAD_LEFT);
        $start_date = "$year-$month_str-01";
        $end_date = date("Y-m-t", strtotime($start_date));
        
        $working_days = 0;
        $current = strtotime($start_date);
        $end = strtotime($end_date);
        
        while ($current <= $end) {
            $day_of_week = date('w', $current);
            if ($day_of_week != 0) {
                $working_days++;
            }
            $current = strtotime('+1 day', $current);
        }
        
        $stmt = $conn->prepare("
            SELECT 
                sm.enrollno,
                COUNT(DISTINCT DATE(al.scan_time)) as present_days
            FROM student_master sm
            LEFT JOIN att_log al ON sm.enrollno = al.enrollno 
                AND DATE(al.scan_time) >= ? 
                AND DATE(al.scan_time) <= ?
            WHERE sm.institute_code = ? 
            AND sm.Course_Code = ? 
            AND sm.Year_Code = ? 
            AND sm.Div_code = ?
            GROUP BY sm.enrollno
        ");
        $stmt->bind_param("ssssss", $start_date, $end_date, $institute_code, $course_code, $year_code, $div_code);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $total_percentage = 0;
        $student_count = 0;
        
        while ($row = $result->fetch_assoc()) {
            $present_days = $row['present_days'];
            $percentage = $working_days > 0 ? ($present_days / $working_days) * 100 : 0;
            $total_percentage += $percentage;
            $student_count++;
        }
        $stmt->close();
        
        $average_attendance = $student_count > 0 ? round($total_percentage / $student_count, 2) : 0;
        
        $monthwise_data[] = [
            'month' => $month,
            'month_name' => date('F', mktime(0, 0, 0, $month, 1)),
            'working_days' => $working_days,
            'average_attendance' => $average_attendance
        ];
    }
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'year' => $year,
            'total_students' => $total_students,
            'monthwise_history' => $monthwise_data
        ]
    ]);
}

function getStudentsList($conn, $input) {
    $institute_code = $input['institute_code'] ?? '';
    $course_code = $input['course_code'] ?? '';
    $year_code = $input['year_code'] ?? '';
    $div_code = $input['div_code'] ?? '';
    
    if (!$course_code || !$year_code || !$div_code) {
        throw new Exception('Course, year, and division codes are required');
    }
    
    $stmt = $conn->prepare("
        SELECT enrollno, roll_no, Name, email, contact
        FROM student_master
        WHERE institute_code = ? 
        AND Course_Code = ? 
        AND Year_Code = ? 
        AND Div_code = ?
        ORDER BY roll_no ASC
    ");
    $stmt->bind_param("ssss", $institute_code, $course_code, $year_code, $div_code);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $students = [];
    while ($row = $result->fetch_assoc()) {
        $students[] = [
            'enrollno' => $row['enrollno'],
            'roll_no' => $row['roll_no'],
            'name' => $row['Name'],
            'email' => $row['email'],
            'contact' => $row['contact']
        ];
    }
    $stmt->close();
    
    echo json_encode([
        'status' => 'success',
        'students' => $students
    ]);
}

function getCourses($conn, $institute_code) {
    $stmt = $conn->prepare("
        SELECT id, course_code, course_name
        FROM courses
        WHERE institute_code = ? AND status = 1
        ORDER BY course_name ASC
    ");
    $stmt->bind_param("s", $institute_code);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $courses = [];
    while ($row = $result->fetch_assoc()) {
        $courses[] = [
            'id' => $row['id'],
            'course_code' => $row['course_code'],
            'course_name' => $row['course_name']
        ];
    }
    $stmt->close();
    
    echo json_encode([
        'status' => 'success',
        'courses' => $courses
    ]);
}

function getYears($conn, $input) {
    $course_id = $input['course_id'] ?? '';
    
    if (!$course_id) {
        throw new Exception('Course ID is required');
    }
    
    $stmt = $conn->prepare("
        SELECT id, year_code, year_name, year_number
        FROM course_years
        WHERE course_id = ? AND status = 1
        ORDER BY year_number ASC
    ");
    $stmt->bind_param("s", $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $years = [];
    while ($row = $result->fetch_assoc()) {
        $years[] = [
            'id' => $row['id'],
            'year_code' => $row['year_code'],
            'year_name' => $row['year_name'],
            'year_number' => $row['year_number']
        ];
    }
    $stmt->close();
    
    echo json_encode([
        'status' => 'success',
        'years' => $years
    ]);
}

function getDivisions($conn, $input) {
    $course_year_id = $input['course_year_id'] ?? '';
    
    if (!$course_year_id) {
        throw new Exception('Course year ID is required');
    }
    
    $stmt = $conn->prepare("
        SELECT id, division_code, division_name
        FROM divisions
        WHERE course_year_id = ? AND status = 1
        ORDER BY division_name ASC
    ");
    $stmt->bind_param("s", $course_year_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $divisions = [];
    while ($row = $result->fetch_assoc()) {
        $divisions[] = [
            'id' => $row['id'],
            'division_code' => $row['division_code'],
            'division_name' => $row['division_name']
        ];
    }
    $stmt->close();
    
    echo json_encode([
        'status' => 'success',
        'divisions' => $divisions
    ]);
}
?>