<?php
header('Content-Type: application/json');
require 'connection.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode([
        'status' => 'error',
        'message' => 'Only GET method allowed'
    ]);
    exit;
}

$client_id   = $_GET['client_id'] ?? '';
$course_id   = $_GET['course_id'] ?? '';
$year_id     = $_GET['year_id'] ?? '';
$division_id = $_GET['division_id'] ?? '';
$enrollment_no  = $_GET['enrollment_no'] ?? '';

if (!$client_id || !$course_id || !$year_id || !$division_id || !$enrollment_no) {
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => 'Missing required parameters'
    ]);
    exit;
}

$stmt = $conn->prepare("
    SELECT id, student_uid, name, gender, contact, email, address
    FROM uni_stud_data
    WHERE enrollment_no = ? AND client_id = ? AND status = 'active'
    LIMIT 1
");
$stmt->bind_param("si", $enrollment_no, $client_id);
$stmt->execute();
$student = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$student) {
    http_response_code(404);
    echo json_encode([
        'status' => 'error',
        'message' => 'Student not found'
    ]);
    exit;
}

$student_id = $student['id'];

$stmt = $conn->prepare("
    SELECT roll_no 
    FROM uni_student_academic 
    WHERE student_id = ? 
      AND year_id = ? 
      AND division_id = ? 
      AND client_id = ? 
      AND status = 'active'
    LIMIT 1
");
$stmt->bind_param("issi", $student_id, $year_id, $division_id, $client_id);
$stmt->execute();
$rollRow = $stmt->get_result()->fetch_assoc();
$stmt->close();

$student['roll_no'] = $rollRow['roll_no'] ?? '';

$stmt = $conn->prepare("
    SELECT COUNT(*) AS total,
           SUM(CASE WHEN status = 'P' THEN 1 ELSE 0 END) AS present
    FROM uni_class_attend_log
    WHERE student_id = ? AND client_id = ?
");
$stmt->bind_param("ii", $student_id, $client_id);
$stmt->execute();
$row = $stmt->get_result()->fetch_assoc();
$stmt->close();

$total   = (int)$row['total'];
$present = (int)$row['present'];
$overallPercentage = $total > 0 ? round(($present / $total) * 100, 2) : 0;

$months = [];

$stmt = $conn->prepare("
    SELECT attendance_date, status, scantime
    FROM uni_class_attend_log
    WHERE student_id = ? AND client_id = ?
    ORDER BY attendance_date DESC
");
$stmt->bind_param("ii", $student_id, $client_id);
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    $monthKey = date("F Y", strtotime($row['attendance_date']));

    if (!isset($months[$monthKey])) {
        $months[$monthKey] = [
            'present' => 0,
            'total' => 0,
            'percentage' => 0,
            'history' => []
        ];
    }

    $months[$monthKey]['total']++;
    if ($row['status'] === 'P') {
        $months[$monthKey]['present']++;
    }

    $months[$monthKey]['history'][] = [
        'attendance_date' => $row['attendance_date'],
        'status' => $row['status'],
        'scan_time' => $row['scantime']
    ];
}

foreach ($months as $key => $m) {
    $months[$key]['percentage'] = $m['total'] > 0
        ? round(($m['present'] / $m['total']) * 100, 2)
        : 0;
}

$conn->close();

echo json_encode([
    'status' => 'success',
    'student' => $student,
    'report' => [
        'overall_percentage' => $overallPercentage,
        'months' => (object)$months
    ]
]);
